local StatusEffect = require "util/statuseffects/statuseffect"

local Frozen = Class(StatusEffect)

function Frozen:construct(object, properties)
  StatusEffect.construct(self, "Frozen")

  self.duration = properties.duration

  self.fx = nil

  self.co = Co.create(function()
    local transform = object.transform

    local health = object.health
    local oldColor = nil

    if not health.isAlive then
      return
    end

    if object.skeletonRenderer then
      oldColor = object.skeletonRenderer.tintColor
      object.skeletonRenderer.tintColor = Color(0.2, 0.5, 0.95, 1.0)
      object.skeletonRenderer.timeScale = 0.0
    end

    if object.soundEmitter then
      object.soundEmitter:postEvent("GetFrozen")
    end

    if self.fx then
      self.fx.object:destroy()
    end

    self.fx = CreateObject("FrozenFX", object.transform.worldPosition - Vector(0.0, 1.0)):addComponent("FrozenFX")

    local lastPos = transform.worldPosition

    while self.duration > 0.0 do
      if object.skeletonRenderer then
        object.skeletonRenderer.timeScale = 0.0
      end

      self.duration = self.duration - Time.fixedDeltaTime
      transform.localPosition = lastPos

      if object.health and not object.health.isAlive then
        self:dispose()
        return
      end

      Co.yield()
    end

    if object.soundEmitter then
      object.soundEmitter:postEvent("GetUnfrozen")
    end

    if object.skeletonRenderer then
      local color = object.skeletonRenderer.tintColor

      local t = 0.0
      while t < 1.0 do
        t = t + Time.fixedDeltaTime
        object.skeletonRenderer.tintColor = Color.mix(color, Color.white, t)
        self.fx.skeletonRenderer.tintColor = Color(1.0, 1.0, 1.0, 1.0 - t)
        Co.yield()
      end

      self.fx.object:destroy()
      self.fx = nil

      if health.isAlive and object.skeletonRenderer then   
        object.skeletonRenderer.tintColor = oldColor
        object.skeletonRenderer.timeScale = 1.0
      end
    end

    self.ended = true
  end)
end

function Frozen:run()
  if not self.co or not self.co:update() then
    self.co = nil
    
    if self.fx then
      self.fx.object:destroy()
      self.fx = nil
    end

    return false
  end

  return true
end

function Frozen:stackWith(other)
  if self.ended then
    return
  end

  self.duration = self.duration + other.duration
end

function Frozen:dispose()
  if self.fx then
    self.fx.object:destroy()
    self.fx = nil
  end
end

return Frozen
