#define PI 3.1415926
// uniform float globalTime;

uniform bool lightBuffers;
uniform samplerBuffer lightValues;
uniform isamplerBuffer lightKeys;
uniform int lightDimsX;
uniform int lightDimsZ;
uniform float lightCellSize;


//
//
// handy value clamping to 0 - 1 range
float saturate(in float value)
{
	return clamp(value, 0.0, 1.0);
}

vec3 saturate(in vec3 value)
{
	return clamp(value, vec3(0.0), vec3(1.0));
}

// phong (lambertian) diffuse term
float phong_diffuse()
{
	return (1.0 / PI);
}



// following functions are copies of UE4
// for computing cook-torrance specular lighting terms

float D_blinn(in float roughness, in float NdH)
{
	float m = roughness * roughness;
	float m2 = m * m;
	float n = 2.0 / m2 - 2.0;
	return (n + 2.0) / (2.0 * PI) * pow(NdH, n);
}

float D_beckmann(in float roughness, in float NdH)
{
	float m = roughness * roughness;
	float m2 = m * m;
	float NdH2 = NdH * NdH;
	return exp((NdH2 - 1.0) / (m2 * NdH2)) / (PI * m2 * NdH2 * NdH2);
}

float D_GGX(in float roughness, in float NdH)
{
	float m = roughness * roughness;
	float m2 = m * m;
	float d = (NdH * m2 - NdH) * NdH + 1.0;
	return m2 / (PI * d * d);
}

float GTR1(in float NdH, in float a)
{
	if ( a >= 1 ) return 1.0 / PI;
	float a2 = a * a;
	float t = 1 + (a2 - 1) * NdH * NdH;
	return (a2-1) / (PI*log(a2)*t);
}

float G_schlick(in float roughness, in float NdV, in float NdL)
{
	float k = roughness * roughness * 0.5;
	float V = NdV * (1.0 - k) + k;
	float L = NdL * (1.0 - k) + k;
	return 0.25 / (V * L);
}


// simple phong specular calculation with normalization
vec3 phong_specular(in vec3 V, in vec3 L, in vec3 N, in vec3 specular, in float roughness)
{
	vec3 R = reflect(-L, N);
	float spec = max(0.0, dot(V, R));

	float k = 1.999 / (roughness * roughness);

	return min(1.0, 3.0 * 0.0398 * k) * pow(spec, min(10000.0, k)) * specular;
}

// simple blinn specular calculation with normalization
float blinn_specular(in float NdH, in float specular, in float roughness)
{
	float k = 1.999 / (roughness * roughness);

	return min(1.0, 3.0 * 0.0398 * k) * pow(NdH, min(10000.0, k)) * specular;
}

// cook-torrance specular calculation
float cooktorrance_specular(in float NdL, in float NdV, in float NdH, in float HdV, in float specular, in float roughness)
{
	NdL = saturate(NdL);
	NdV = saturate(NdV);
	NdV = saturate(NdV);
	NdH = saturate(NdH);
	specular = saturate(specular);
	roughness = saturate(roughness);
#define COOK_GGX
#ifdef COOK_BLINN
	float D = D_blinn(roughness, NdH);
#endif

#ifdef COOK_BECKMANN
	float D = D_beckmann(roughness, NdH);
#endif

#ifdef COOK_GGX
	float D = D_GGX(roughness, NdH);
#endif

	// float x = 2.0 * NdH / HdV;
	// float G = min( 1.0, min(x * NdV, x * NdL) );
	float G = G_schlick(0.6, NdV, NdL);

	float rimFactor = 0.9; // how much rim lighting do we want?
	float rim = 1.0;//mix(1.0 - roughness * rimFactor * 0.9, 1.0, NdV);

	return (1.0 / rim) * specular * G * D;
}

float orennayar_diffuse( in vec3 light, in vec3 view, in vec3 normal, in float roughness )
{
	float VdN = dot(view,normal);
	float LdN = dot(light,normal);
	float cos_theta_i = LdN;
	float theta_r = acos( VdN );
	float theta_i = acos( cos_theta_i );
	float cos_phi_diff = dot( normalize( view - normal * VdN ),
			normalize( light - normal * LdN ) );
	float alpha = max( theta_r, theta_i );
	float beta = min( theta_r, theta_i );
	float sigma2 = roughness * roughness;
	float A = 1.0 - 0.5 * sigma2 / (sigma2 + 0.33);
	float B = 0.45 * sigma2 / (sigma2 + 0.09);

	return saturate( cos_theta_i ) *
		(A + (B * saturate( cos_phi_diff ) * sin(alpha) * tan(beta)));
}

// compute fresnel specular factor for given base specular and product
// product could be NdV or VdH depending on used technique
float fresnel_factor(in float f0, in float product)
{
	return mix(f0, 1.0, pow(1.0 - product, 5.0));
}

float F_schlick(in float f0, in float fd90, in float u)
{
	/* return mix(f0, fd90, pow(1.01 - u, 5.0)); */
	/* return f0 + (fd90 - f0) * pow(1.0 - u, 5.0); */
	return f0 + (fd90 - f0) * pow(1.0 - u, 5.0);
}

float SchlickFresnel(float u)
{
	float m = clamp(1-u, 0, 1);
	float m2 = m*m;
	return m2*m2*m;
}

float disney_diffuse( in float NdV, in float NdL, in float LdH, in float roughness )
{
#if 1
	// Disney's Diffuse BRDF on its own
	float FL = SchlickFresnel(NdL);
	float FV = SchlickFresnel(NdV);
	float Fd90 = 0.5 + (2.0 * LdH*LdH * roughness);
	float Fd = mix(1.0, Fd90, FL) * mix(1.0, Fd90, FV);
	return Fd;
#else
	// Disney's Diffuse BRDF, with EA's energy renomalisation adaptation.
	float energyBias = 0.5;//mix(0, 0.5, roughness);
	float energyFactor = mix(1.0, 1.0 / 1.51, roughness);
	float fd90 = energyBias + 2.0 * LdH * LdH * roughness;
	const float f0 = 1.0;
	float lightScatter = F_schlick( f0, fd90, NdL );
	float viewScatter = F_schlick( f0, fd90, NdV);
	float Fd = lightScatter * viewScatter * energyFactor;
	return Fd;
#endif
}


// vec3 env(vec3 p)
// {
// 	p *= 2.0;
// 	p.xz += globalTime*0.5;
// 	float n3D2 = noise(p*2.0);
// 	float c = noise(p) *0.57 + n3D2 * 0.28 + noise(p*4.0) * 0.15;
// 	c = smoothstep(0.5, 1.0, c);
// 	p = vec3(c*c*c*c,c*c,c); // blue-ish
// 	return mix(p.zxy, p, n3D2*0.34 + 0.665); // bit of purple
// }


float lambert_diffuse( in float NdL )
{
	return NdL;
}

void sRGB2Linear( inout vec3 a )
{
	a.rgb *= a.rgb;
}

void Linear2sRGB( inout vec3 a )
{
	a.rgb = sqrt(a.rgb);
	// a.rgb = vec3(1);
}

float smithG_GGX(float NdV, float alphaG)
{
	float a = alphaG*alphaG;
	float b = NdV*NdV;
	return 1.0 / (NdV + sqrt(a+b-(a*b)));
}

float sqr(float x)
{
	return x * x;
}

float smithG_GGX_aniso(float NdotV, float VdotX, float VdotY, float ax, float ay)
{
	return 1.0 / (NdotV + sqrt( sqr(VdotX*ax) + sqr(VdotY*ay) + sqr(NdotV) ));
}

float GTR2_aniso(float NdotH, float HdotX, float HdotY, float ax, float ay)
{
    return 1 / (PI * ax*ay * sqr( sqr(HdotX/ax) + sqr(HdotY/ay) + NdotH*NdotH ));
}

vec3 ambient_brdf(vec3 color)
{
	vec3 dF = lightSource[0].ambient.rgb * color;
	return dF;
}

float V_SmithGGXCorrelated( float NdV, float NdL, float specRoughness)
{
	float a2 = specRoughness * specRoughness;
	float GGXV = NdL * sqrt(NdV * NdV * (1.0 - a2) + a2);
	float GGXL = NdV * sqrt(NdL * NdL * (1.0 - a2) + a2);
	return 0.5 / (GGXV + GGXL);
}

vec3 brdf(vec3 color, vec3 lightColor, vec3 N, vec3 L, vec3 V, vec3 H,
		float specular, float rimSpecular, float roughness, float specRoughness, float subsurface, float clearcoatAmt )
{
	float actualNdL = (dot(N, L));
	float NdL = 0.5f + (0.5f * actualNdL);
	float NdV = (dot(N, V));

	if ( NdL < 0 )
		return vec3(0);

	float NdH = (dot(N, H));
	float LdH = (dot(L, H));
	float HdV = (dot(H, V));
	float LdV = (dot(L, V));

	vec3 result = vec3(0);
	// if ( NdL <= 0.0 )// || NdV <= 0.0 )
	// 	return vec3(0);

	// float linearRoughness = sqrt(roughness);
	float linearRoughness = roughness * roughness;//sqrt(roughness);
	// if ( NdL > 0.0 ) //&& NdV > 0.0 )
	{

// #define D_NONE
#define D_DISNEY
// #define D_LAMBERT
// #define D_ORENNAYAR

// #define S_NONE
#define S_COOKTORRANCE
// #define S_BLINN
// #define S_DISNEY

		// Diffuse
		//
#ifdef D_NONE
		float diffref = 0.0;
#endif
#ifdef D_LAMBERT
		float diffref = lambert_diffuse( NdL ) / PI;
#endif
#ifdef D_ORENNAYAR
		float diffref = orennayar_diffuse( L, V, N, roughness) / PI;
#endif
#ifdef D_DISNEY
		float diffref = disney_diffuse( NdV, NdL, LdH, linearRoughness) / PI;
#endif

		// Specular
		//
		float specfresnel = F_schlick(specular, rimSpecular, NdV);
#ifdef S_NONE
		float specref = 0.0;
#endif
#ifdef S_BLINN
		float specref = blinn_specular(NdH, specfresnel, specRoughness);
#endif
#ifdef S_COOKTORRANCE
		float specref = cooktorrance_specular(NdL, NdV, NdH, HdV, specfresnel, specRoughness) / PI;
#endif
		float FH = SchlickFresnel(LdH);
		// float FH = F_schlick(0, 1, LdH);
#ifdef S_DISNEY

		vec3 X = normalize(cross(N,vec3(1.0,0.0,0.0)));
		vec3 Y = normalize(cross(N,X));

		float anisotropic = 0.0;
		float aspect = sqrt(1-anisotropic*0.9);//sqrt(1.0);
		float ax = max(0.001, (specRoughness * specRoughness)/aspect);
		float ay = max(0.001, (specRoughness * specRoughness)*aspect);
		float Ds = GTR2_aniso(NdH, dot(H,X), dot(H,Y), ax, ay);
		// float FH = SchlickFresnel(LdH);
		// vec3 Fs = mix(Cspec0, vec3(1), FH);
		vec3 Fs = vec3(1);
		float Gs = smithG_GGX_aniso(NdL, dot(L,X), dot(L,Y), ax, ay);
		Gs *= smithG_GGX_aniso(NdV, dot(V,X), dot(V,Y), ax, ay);

		float specref = (Gs * Ds) / PI;
		// float specref = (Ds * Gs) / PI;//(Gs * Ds) / PI;

		// float fresnel0 = 0.0;
		// float f90 = saturate (50.0 * fresnel0 * 0.33 ) ;
		// float F = F_schlick( fresnel0, f90, LdH );
		// float Vis = V_SmithGGXCorrelated(NdV, NdL, specRoughness * 0.01);
		// float D = D_GGX(NdH,specRoughness);
		// float specref = D * F * Vis / PI;
#endif
		// specref *= NdL;

#define DISNEY_SUBSURFACE
		// #define EA_SUBSURFACE

#ifdef DISNEY_SUBSURFACE
		// Subsurface;  Disney-style
		float FL = saturate(F_schlick(0,1,NdL));
		float FV = saturate(F_schlick(0,1,NdV));
		float Fss90 = LdH * LdH * linearRoughness;
		float Fss = mix(1.0, Fss90, FL) * mix(1.0, Fss90, FV);
		float ss = 1.25 * saturate((Fss * (1 / (NdL + NdV))) - 0.5) + 0.5;
		ss = ss / PI;

		diffref = mix(diffref, ss, subsurface);
#endif // DISNEY_SUBSURFACE

		vec3 dF = diffref * lightColor * NdL;
		vec3 dS = specref * lightColor;// * NdL;

		vec3 sheen = vec3(0);
		vec3 clearcoat = vec3(0);

		// if ( NdL > 0.0 )
		{
#define SHEEN
#ifdef SHEEN
		// sheen, woo!
		sheen = vec3(FH) * clearcoatAmt * lightColor * saturate(actualNdL);
#endif // SHEEN


		// [TODO]:  We need an explicit parameter to control Clearcoat.  Looks
		// good on most stuff, but we definitely need to not have it applied to
		// the terrain!
		//
#define CLEARCOAT
#ifdef CLEARCOAT
		// clearcoat, woo!
		float clearcoatGloss = 0.8;
		float Dr = GTR1(NdH, mix(0.1, 0.001, clearcoatGloss));
		float Fr = mix(0.04, 1.0, FH);
		float Gr = smithG_GGX(NdL, 0.25) * smithG_GGX(NdV, 0.25);
		clearcoat = vec3( 0.25 * clearcoatGloss * Gr * Fr * Dr ) * 1.0 * clearcoatAmt * lightColor;
#endif // CLEARCOAT
		}

		result += (dF * color) + dS + sheen + clearcoat;
	}

	// DEBUGGING
	/* vec3 newColor = (0.5*N) + 0.5; */
	/* color.rgb = mix(color.rgb, newColor, 0.99); */


#ifdef EA_SUBSURFACE
	// Subsurface;  EA-style
	// Based upon "Approximating Translucency for a fast cheap and convincing
	// subsurface scattering look"
	float fLTDistortion = 0.1; // distorts the normal
	int iLTPower = 8;
	float fLTAmbient = 0.0; // amount of constant ambient transparency
	float fLTThickness = 0.5; // high values are LESS THICK, because EA knows how to name variables. :)
	float fLTScale = 1.0;
	vec3 lightAttenuation = vec3(1.0);

	vec3 vLTLight = L + (N * fLTDistortion);
	float fLTDot = pow(saturate(dot(V,-vLTLight)), iLTPower) * fLTScale;
	float fLT = (fLTDot + fLTAmbient) * fLTThickness;

	result += vec3(0.3,0.3,0.7) * color * lightColor * fLT;//color * lightColor * fLT; //vec3(1,0,0) * color * /* lightColor */ fLT;
#endif // EA_SUBSURFACE

	return result;
}

vec3 do_point_light(int lightId, vec3 color,
		vec3 N, vec3 V,
		float specular, float rimSpecular, float roughness, float specRoughness, float subsurface, float clearcoat)
{
	vec3 result = vec3(0);
	int valueIndex = lightId * 2;
	vec4 posRadius = texelFetch(lightValues, valueIndex+0).rgba;
	float radiusSq = posRadius.w * posRadius.w;
	vec3 worldDelta = posRadius.xyz - worldPosition.xyz;
	float d = dot(worldDelta,worldDelta);

	// result.b = 1.0;

	if ( d <= 1.0 * radiusSq )
	{
		// vec3 viewPos = (worldToView * vec4(posRadius.xyz, 1.0)).xyz;
		vec3 lightColor = texelFetch(lightValues, valueIndex+1).rgb;

		vec3 lightDirectionHere = worldDelta;
		if ( d > 0.0 )
			lightDirectionHere = normalize(lightDirectionHere);
		vec3 viewLightDirectionHere = (worldToView * vec4(lightDirectionHere, 0.0)).xyz;

		// drop brightness with distance squared.
		//
		float dist = sqrt(d);

		d = dist;
		d = d / posRadius.w; // 0 .. 1 as we go center->outside of light sphere
		d = 1.0 - d;         // 1 .. 0 as we go center->outside of light spehre
		d = d*d;             // Apply ease-in
		d *= 10.0;           // Multiply by 10 because PBL and lights need to be brighter.

		vec3 PL = viewLightDirectionHere;//normalize(viewPos - V);
		vec3 PH = normalize(PL + V);

		float NdL = saturate(dot(N,PL)); // modulate color down as NdL drops.
		// NdL = smoothstep(0.0, 1.0, NdL);

		// result.rgb += lightColor * d * NdL;
		// if ( NdL > 0.0 )
		{
			result.rgb = brdf(color, lightColor * d, N, PL, V, PH,
					specular, rimSpecular, roughness, specRoughness, subsurface, clearcoat);
		}

		// result.rgb = mix(vec3(0,1,0), result.rgb, dist / posRadius.w);

		// result.rgb = brdf(color, lightColor * d, N, PL, V, PH, 1.0,
		// 		1.0, 0.0, 0.0, 0.0);
		//
	// result.rgb = mix( saturate(PL), result.rgb, 0.05 );
		// result.rgb = vec3(saturate(dot(PH,PL)));
	}
	return result;
}

vec3 Uncharted2Curve( vec3 light )
{
	float A = 0.15;
	float B = 0.50;
	float C = 0.10;
	float D = 0.20;
	float E = 0.02;
	float F = 0.30;
	return ((light*(A*light+C*B)+D*E)/(light*(A*light+B)+D*F))-E/F;
}

vec3 Uncharted2CurveMap( vec3 color )
{
	float W = 11.2;
	float exposureBias = 1.0;
	vec3 curr = Uncharted2Curve(exposureBias * color);
	vec3 whiteScale = 1.0 / Uncharted2Curve(vec3(W));
	return curr * whiteScale;

}

vec3 LumUncharted2Curve( vec3 light )
{
	float A = 0.15;
	float B = 0.50;
	float C = 0.10;
	float D = 0.20;
	float E = 0.02;
	float F = 0.30;
	float W = 11.2;
	float lum = dot(light.xyz, vec3(0.2989, 0.5870, 0.1140));
	// float lum = max( dot(light.xyz, vec3(0.2126, 0.7152, 0.0722)), 0.001 );
	float newLum = ((lum*(A*lum+C*B)+D*E)/(lum*(A*lum+B)+D*F))-E/F;
	float scale = newLum / lum;

	return light * scale;
}

vec3 AcesCurve( vec3 light )
{
	float a = 2.51;
    float b = 0.03;
    float c = 2.43;
    float d = 0.59;
    float e = 0.14;
	light *= 0.25;
    return clamp((light*(a*light+b))/(light*(c*light+d)+e),0.0,1.0);
}

vec3 LumAcesCurve( vec3 light )
{
	float tA = 2.51f;
    float tB = 0.03f;
    float tC = 2.43f;
    float tD = 0.59f;
    float tE = 0.14f;

	// exposure, to approximately match brightness to game
	light *= 0.25f;

	float lum = max( dot(light.xyz, vec3(0.2126, 0.7152, 0.0722)), 0.001 );
	float newLum = clamp((lum*(tA*lum+tB))/(lum*(tC*lum+tD)+tE),0.0,1.0);
	float scale = newLum / lum;

    return light * scale;
}

const mat3 ACESInputMat = mat3(
    vec3(0.59719, 0.35458, 0.04823),
    vec3(0.07600, 0.90834, 0.01566),
    vec3(0.02840, 0.13383, 0.83777)
);

// ODT_SAT => XYZ => D60_2_D65 => sRGB
const mat3 ACESOutputMat = mat3(
    vec3( 1.60475, -0.53108, -0.07367),
    vec3(-0.10208,  1.10813, -0.00605),
    vec3(-0.00327, -0.07276,  1.07602)
);

vec3 RRTAndODTFit(vec3 v)
{
    vec3 a = v * (v + 0.0245786f) - 0.000090537f;
    vec3 b = v * (0.983729f * v + 0.4329510f) + 0.238081f;
    return a / b;
}


vec3 AcesFitted( vec3 color )
{
	float exposure = 0.4;
	color *= exposure;
	color = ACESInputMat * color;
	// apply RRT and ODT
	color = RRTAndODTFit(color);
	color = ACESOutputMat * color;

	// clamp to (0..1)
	color = saturate(color);
	return color;
}

vec3 LumReinhardCurve( vec3 light )
{
	float lum = dot(light.xyz, vec3(0.2989, 0.5870, 0.1140));
	float newLum = lum/(1+lum);
	return light / lum * newLum;
}
vec3 RgbReinhardCurve( vec3 light )
{
	return light/(1+light);
}
vec3 HybridReinhardCurve( vec3 light )
{
	float vividness = 0.0;
	vec3 lum = LumReinhardCurve(light);
	vec3 rgb = RgbReinhardCurve(light);
	return mix(rgb, lum, vividness);
}
vec3 LumLogCurve( vec3 light )
{
	float crush = 0.0000001;
	float fstops = 2.0;
	float lum = dot(light.xyz, vec3(0.2989, 0.5870, 0.1140));
	return smoothstep(crush, fstops, light.xyz * log2(1+lum)/lum);
}
vec3 RgbLogCurve( vec3 light )
{
	float crush = 0.0000001;
	float fstops = 2.0;
	return smoothstep(crush, fstops, log2(1+light.xyz));
}

vec3 shade(vec3 color, out vec3 linearColor,
		vec3 N, vec3 L, vec3 V, vec3 H, float illuminated,
		float specular, float rimSpecular, float roughness, float specRoughness,
		float subsurface, float clearcoat, vec3 fogColor, float fogAmt, bool doPointLights)
{
	sRGB2Linear(color.rgb);
	vec3 result = ambient_brdf(color);
	if ( illuminated > 0.0 )
	{
		vec3 lightColor = lightSource[0].diffuse.rgb * illuminated * PI;
		result += brdf(color, lightColor, N, L, V, H,
				specular, rimSpecular, roughness, specRoughness, subsurface, clearcoat);
	}

	// Now let's do our point lights.
	if ( doPointLights && lightBuffers && lightDimsX > 0 )
	{
		int xCell = clamp(int(floor(worldPosition.x / lightCellSize)), 0, lightDimsX-1);
		int zCell = clamp(int(floor(worldPosition.z / lightCellSize)), 0, lightDimsZ-1);
		int cellIndex = (zCell * lightDimsX) + xCell;
		int blockIndex = cellIndex;

		int lightCount = 0;
		while ( blockIndex >= 0 )
		{
			ivec4 lightBlock = texelFetch(lightKeys, blockIndex).rgba;

			if ( lightBlock.r < 0 )
				break;
			result.rgb += do_point_light( lightBlock.r, color, N, V, specular, rimSpecular, roughness, specRoughness, subsurface, clearcoat );

			if ( lightBlock.g < 0 )
				break;
			result.rgb += do_point_light( lightBlock.g, color, N, V, specular, rimSpecular, roughness, specRoughness, subsurface, clearcoat );

			if ( lightBlock.b < 0 )
				break;
			result.rgb += do_point_light( lightBlock.b, color, N, V, specular, rimSpecular, roughness, specRoughness, subsurface, clearcoat );

			blockIndex = lightBlock.a;

			lightCount+=3;
		}
	}

	// Finally, fog.
	//
	vec3 linFogColor = fogColor;
	sRGB2Linear(linFogColor);

	linearColor = result;

	// And tone mapping.
	// result = RgbReinhardCurve(result);
	// result = LumReinhardCurve(result);
	// result = Uncharted2Curve(result);
	// result = Uncharted2CurveMap(result);
	// result = LumUncharted2Curve(result);
	// result = AcesCurve(result);
	// result = LumAcesCurve(result);

	vec3 filmic = AcesFitted(result);
	vec3 lumic = LumAcesCurve(result);

	result = mix(filmic,lumic,0.5);

	// result = LumLogCurve(result);
	// result = RgbLogCurve(result);

	// extra contrast?  Sure, have some!
	// result = mix(result,smoothstep(vec3(0),vec3(1),result),0.8);

	result = mix( result, linFogColor, fogAmt );
	Linear2sRGB(result);

	return result;
}


